#!/usr/bin/env Rscript

suppressPackageStartupMessages(require(optparse))
suppressPackageStartupMessages(require(tidyverse))

read_survey <- function(path) {
  data = read_csv(path, col_types=cols_only(
    issue = col_character(),
    category = col_integer(),
    tool = col_character(),
    question = col_character(),
    participant = col_character(),
    response = col_factor(levels=c("1", "2", "3", "4", "5"))
  ))
  
  data |> mutate(
    response = fct_recode(response, 
                          "Strongly disagree" = "1", 
                          "Disagree" = "2", 
                          "Neither disagree nor agree" = "3", 
                          "Agree" = "4", 
                          "Strongly agree" = "5")
  )
}

internal = read_survey("internal.csv") |> mutate(type = "internal")
external = read_survey("external.csv") |> mutate(type = "external")

data = bind_rows(internal, external) |>
  group_by(type, category, tool, question, response) |>
  count()

plot = ggplot(data, aes(fill = response, y = fct_rev(question), x = n)) +
  geom_bar(position = "fill", stat = "identity") +
  guides(fill = guide_legend(reverse = TRUE)) +
  scale_fill_brewer(palette = "RdBu", name = "Response") +
  facet_wrap(
    vars(tool),
  ) +
  scale_x_continuous(labels = scales::percent) +
  theme_bw(12) +
  theme(
    axis.text.x = element_text(angle = 90, vjust = 0.5, hjust=1),
    legend.position = "bottom",
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
  ) +
  labs(x = "Percentage of responses", y = "Statement")

ggsave(plot, file = "responses.pdf", width = 8.5, height = 3)
